#ifndef _GAMELOOP_H__
#define _GAMELOOP_H__

#include <list>
#include <vector>
#include "sprig.h"
#include "Player.h"
#include "Level.h"
#include "Libraries/PL_Loop.h"

using namespace PL;
using namespace std;

class Player;

class TerrainMap
{
    public:
    int width, height;
    int gridW, gridH;
    int* grid;
    
    TerrainMap(int width, int height)
        : width(width)
        , height(height)
        , gridW(100)
        , gridH(100)
    {
        grid = new int[gridW*gridH];
        for(int i = 0; i < gridW*gridH; i++)
        {
            grid[i] = 0;
        }
    }
    
    int getValue(float x, float y)
    {
        if(x < 0 || x > width
           || y < 0 || y > height)
        return 0;
        
        int ix = int(x/width * (gridW-1));
        int iy = int(y/height * (gridH-1));
        
        return grid[ix + iy * gridW];
    }
    
    int getValueAbs(int x, int y)
    {
        if(x < 0 || x >= gridW
           || y < 0 || y >= gridH)
        return 0;
        
        return grid[x + y * gridW];
    }
    
    void set(float x, float y, int value)
    {
        if(x < 0 || x > width
           || y < 0 || y > height)
        return;
        
        int ix = int(x/width * (gridW-1));
        int iy = int(y/height * (gridH-1));
        
        grid[ix + iy * gridW] = value;
    }
    
    void setAbs(int x, int y, int value)
    {
        if(x < 0 || x >= gridW
           || y < 0 || y >= gridH)
        return;
        
        grid[x + y * gridW] = value;
    }
    
    void load(SDL_Surface* map)
    {
        for(int i = 0; i < map->w; i++)
        {
            for(int j = 0; j < map->h; j++)
            {
                Uint32 color = SPG_GetPixel(map, i, j);
                Uint8 r, g, b;
                SDL_GetRGB(color, map->format, &r, &g, &b);
                if(r < 100 && g < 100 && b > 100)  // water
                    set(i, j, 1);
                else if(r > 100 && g > 100 && b < 100)  // mountain
                    set(i, j, 2);
                else if(r < 50 && g < 50 && b < 50)  // void
                    set(i, j, 3);
                else
                    set(i, j, 0);
            }
        }
    }
};

class BurnMap
{
    public:
    int width, height;
    int gridW, gridH;
    float* grid;
    
    BurnMap(int width, int height)
        : width(width)
        , height(height)
        , gridW(100)
        , gridH(100)
    {
        grid = new float[gridW*gridH];
        for(int i = 0; i < gridW*gridH; i++)
        {
            grid[i] = 1.0f;
        }
    }
    
    void set(float x, float y, float value)
    {
        if(x < 0 || x > width
           || y < 0 || y > height)
        return;
        
        int ix = int(x/width * (gridW-1));
        int iy = int(y/height * (gridH-1));
        
        grid[ix + iy * gridW] = value;
    }
    
    void setAbs(int x, int y, float value)
    {
        if(x < 0 || x >= gridW
           || y < 0 || y >= gridH)
        return;
        
        grid[x + y * gridW] = value;
    }
    
    float getValue(float x, float y)
    {
        if(x < 0 || x > width
           || y < 0 || y > height)
        return -10;
        
        int ix = int(x/width * (gridW-1));
        int iy = int(y/height * (gridH-1));
        
        return grid[ix + iy * gridW];
    }
    
    int getValueAbs(int x, int y)
    {
        if(x < 0 || x >= gridW
           || y < 0 || y >= gridH)
        return -10;
        
        return grid[x + y * gridW];
    }
    
    void burn(float x, float y)
    {
        if(x < 0 || x > width
           || y < 0 || y > height)
        return;
        
        int ix = int(x/width * (gridW-1));
        int iy = int(y/height * (gridH-1));
        
        grid[ix + iy * gridW] -= 0.5f;
    }
    
    void burn(float x, float y, float x2, float y2)
    {
        if(x < 0)
            x = 0;
        if(x > width)
            return;
        if(y < 0)
            y = 0;
        if(y > height)
            return;
        if(x2 < 0)
            return;
        if(x2 > width)
            x2 = width;
        if(y2 < 0)
            return;
        if(y2 > height)
            y2 = height;
        
        int ix = int(x/width * (gridW-1));
        int iy = int(y/height * (gridH-1));
        int ix2 = int(x2/width * (gridW-1));
        int iy2 = int(y2/height * (gridH-1));
        
        for(int i = ix; i <= ix2; i++)
        {
            for(int j = iy; j <= iy2; j++)
                grid[i + j * gridW] -= 0.05f;
        }
    }
    
    float getSum()
    {
        float sum = 0;
        for(int i = 0; i < gridW; i++)
        {
            for(int j = 0; j < gridH; j++)
            {
                sum += getValueAbs(i, j);
            }
        }
        return sum;
    }
    
    float getSaved()
    {
        float sum = 0;
        for(int i = 0; i < gridW; i++)
        {
            for(int j = 0; j < gridH; j++)
            {
                float val = getValueAbs(i, j);
                if(val > 0)
                    sum += val;
            }
        }
        return sum;
    }
    
    void load(TerrainMap* terrain)
    {
        for(int i = 0; i < terrain->gridW; i++)
        {
            for(int j = 0; j < terrain->gridH; j++)
            {
                int type = terrain->getValueAbs(i, j);
                if(type == 1 || type == 3)
                    setAbs(i, j, -20);
                else
                    setAbs(i, j, 1);
            }
        }
    }
};

class GameState : public PL_State
{
	public:
	
	int levelNum;
	float nextLevelTimer;
	bool complete;
	
	bool won;
	
	bool bonus;
	
	Player* player;
	Player* bonusPlayer;
	list<Fire> fires;
	list<House> houses;
	
	BurnMap* burnMap;
	TerrainMap* terrainMap;
	
	SDL_Surface* grass;
	
	// Score stuff
	int startingFires;
	list<House> lostHouses;
	bool lostPlayer;
	float lostGrass;
	float savedGrass;
	
	
	GameState()
		: PL_State()
		, levelNum(0)
		, nextLevelTimer(0)
		, complete(false)
		, won(false)
		, bonus(false)
		, player(NULL)
		, bonusPlayer(NULL)
		, burnMap(NULL)
		, grass(NULL)
		, startingFires(0)
		, lostPlayer(false)
		, lostGrass(0)
		, savedGrass(0)
	{
	    
    }
	
	virtual ~GameState()
	{}
    
    void levelOver()
    {
        getScore();
        
        printf("Lost grass: %.3f\n", lostGrass);
        printf("Saved grass: %.3f\n", savedGrass);
        // Show end of level screen here
    }
    
    bool loadNextLevel()
    {
        levelNum++;
        bool result = loadLevel(this, levelNum);
        
        startingFires = fires.size();
        lostHouses.clear();
        lostPlayer = false;
        lostGrass = burnMap->getSaved();
        
        return result;
    }
    
    void getScore()
    {
        if(player->hp < 0)
            lostPlayer = true;
        
        savedGrass = burnMap->getSaved();
        lostGrass -= savedGrass;
    }
    
    void addBackFire(float x, float y)
    {
        Fire f(x, y);
        f.growthFactor = 10;
        f.deathFactor = 10;
        f.speed = 3;
        f.size = 1;
        f.maxSize = 10;
        f.isWild = false;
        fires.push_back(f);
    }
    void addWildFire(float x, float y)
    {
        Fire f(x, y);
        f.growthFactor = 10;
        f.deathFactor = 1;
        f.speed = 7;
        f.size = 10;
        f.isWild = true;
        fires.push_back(f);
    }
    void addFire(Fire* fire)
    {
        fires.push_back(Fire(*fire));
    }
    
    
	
};

class GameUpdate : public PL_Update
{
	public:
	GameUpdate()
		: PL_Update()
	{}
	
	virtual ~GameUpdate()
	{}
	
    virtual void update(PL_State* state);
};

class GameDraw : public PL_Draw
{
	public:
	
	GameDraw()
		: PL_Draw()
	{}
	
	virtual ~GameDraw()
	{}
	
    virtual void draw(PL_State* state);
	
};

class GameEvent : public PL_AnyEvent
{
	public:
	GameEvent()
		: PL_AnyEvent()
	{}
	
	virtual ~GameEvent()
	{}
	
    virtual bool handleEvent(PL_State* state, SDL_Event& event);
	
};











#endif

